/*
 * Decompiled with CFR 0.152.
 */
package dev.wxmc.weatheraddon.warnings;

import dev.wxmc.weatheraddon.warnings.TrackedStorm;
import dev.wxmc.weatheraddon.warnings.WarningLevel;
import java.util.ArrayList;
import java.util.List;
import net.minecraft.ChatFormatting;
import net.minecraft.network.chat.ClickEvent;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.network.chat.Style;
import net.minecraft.network.chat.TextColor;

public class WarningGenerator {
    private static final float DBZ_STATEMENT = 43.0f;
    private static final float DBZ_SVR_WARNING = 52.0f;
    private static final float DBZ_SVR_PDS = 62.0f;
    private static final float DBZ_SVR_DESTRUCTIVE = 65.0f;
    private static final int COLOR_STATEMENT = 16770229;
    private static final int COLOR_SVR = 16753920;
    private static final int COLOR_TORNADO = 0xFF0000;

    public static float calculateIntensity(TrackedStorm storm) {
        int stage = storm.getStage();
        int energy = storm.getEnergy();
        int windspeed = storm.getWindspeed();
        return switch (stage) {
            case 1 -> 0.1f + (float)energy / 100.0f * 0.7f;
            case 2 -> 0.8f + (float)energy / 100.0f * 0.4f;
            case 3 -> 1.2f + (float)windspeed / 100.0f;
            default -> (float)energy / 100.0f * ((float)energy / 100.0f) * 0.1f;
        };
    }

    public static float calculatePeakDBZ(TrackedStorm storm) {
        float intensity = WarningGenerator.calculateIntensity(storm);
        double coreIntensity = 1.5 * Math.sqrt((double)intensity / 2.0);
        double localDBZ = Math.pow(coreIntensity, 0.65);
        return (float)(localDBZ * 60.0);
    }

    public static WarningLevel calculateWarningLevel(TrackedStorm storm) {
        if (storm == null) {
            return WarningLevel.NONE;
        }
        return switch (storm.getStormType()) {
            case 0 -> WarningGenerator.calculateSupercellWarning(storm);
            case 1 -> WarningLevel.NONE;
            case 2 -> WarningLevel.NONE;
            default -> WarningLevel.NONE;
        };
    }

    public static WarningLevel calculateTornadoLevel(TrackedStorm storm) {
        if (storm == null) {
            return WarningLevel.NONE;
        }
        if (storm.getStormType() != 0) {
            return WarningLevel.NONE;
        }
        if (storm.getStage() < 3) {
            return WarningLevel.NONE;
        }
        return WarningGenerator.calculateTornadoWarningLevel(storm);
    }

    public static WarningLevel calculateSevereLevel(TrackedStorm storm) {
        if (storm == null) {
            return WarningLevel.NONE;
        }
        int stormType = storm.getStormType();
        if (stormType != 0) {
            return WarningLevel.NONE;
        }
        float peakDBZ = WarningGenerator.calculatePeakDBZ(storm);
        if (peakDBZ >= 65.0f) {
            return WarningLevel.SVR_DESTRUCTIVE;
        }
        if (peakDBZ >= 62.0f) {
            return WarningLevel.SVR_PDS;
        }
        if (peakDBZ >= 52.0f) {
            return WarningLevel.SVR_WARNING;
        }
        if (peakDBZ >= 43.0f) {
            return WarningLevel.STATEMENT;
        }
        return WarningLevel.NONE;
    }

    private static WarningLevel calculateSupercellWarning(TrackedStorm storm) {
        int stage = storm.getStage();
        if (stage >= 3) {
            return WarningGenerator.calculateTornadoWarningLevel(storm);
        }
        float peakDBZ = WarningGenerator.calculatePeakDBZ(storm);
        if (peakDBZ >= 65.0f) {
            return WarningLevel.SVR_DESTRUCTIVE;
        }
        if (peakDBZ >= 62.0f) {
            return WarningLevel.SVR_PDS;
        }
        if (peakDBZ >= 52.0f) {
            return WarningLevel.SVR_WARNING;
        }
        if (peakDBZ >= 43.0f) {
            return WarningLevel.STATEMENT;
        }
        return WarningLevel.NONE;
    }

    private static WarningLevel calculateTornadoWarningLevel(TrackedStorm storm) {
        int windspeed = storm.getWindspeed();
        float width = storm.getWidth();
        int tornadoOnGroundTicks = storm.getTornadoOnGroundTicks();
        if (windspeed >= 166) {
            return WarningLevel.TOR_EMERGENCY;
        }
        if (windspeed >= 111) {
            return WarningLevel.TOR_PDS;
        }
        if (windspeed >= 40) {
            return WarningLevel.TOR_CONFIRMED;
        }
        if (storm.getTornadoOnGroundTicks() > 0 || windspeed > 0) {
            return WarningLevel.TOR_INDICATED;
        }
        return WarningLevel.TOR_INDICATED;
    }

    private static WarningLevel calculateSquallLineWarning(TrackedStorm storm) {
        return WarningLevel.NONE;
    }

    private static WarningLevel calculateCycloneWarning(TrackedStorm storm) {
        int windspeed = storm.getWindspeed();
        if (windspeed >= 40) {
            return WarningLevel.SVR_DESTRUCTIVE;
        }
        if (windspeed >= 25) {
            return WarningLevel.SVR_WARNING;
        }
        if (windspeed >= 15) {
            return WarningLevel.STATEMENT;
        }
        return WarningLevel.NONE;
    }

    public static String estimateHailSize(TrackedStorm storm) {
        if (!storm.isSupercell()) {
            return "Unknown";
        }
        int stage = storm.getStage();
        int energy = storm.getEnergy();
        return switch (stage) {
            case 1 -> {
                if (energy < 50) {
                    yield "Pea to marble";
                }
                if (energy < 75) {
                    yield "Penny to nickel";
                }
                yield "Quarter";
            }
            case 2 -> {
                if (energy < 34) {
                    yield "Quarter";
                }
                if (energy < 67) {
                    yield "Golf ball";
                }
                yield "Tennis ball";
            }
            case 3 -> "Baseball or larger";
            default -> "None";
        };
    }

    public static boolean shouldMentionTornadoPossible(TrackedStorm storm) {
        return storm.isSupercell() && storm.getStage() == 2 && storm.getMaxStage() >= 3 && storm.getEnergy() >= 34;
    }

    public static boolean shouldMentionTornadoLikely(TrackedStorm storm) {
        return storm.isSupercell() && storm.getStage() == 2 && storm.getMaxStage() >= 3 && storm.getEnergy() >= 67;
    }

    public static List<String> getWarningModifiers(TrackedStorm storm) {
        return new ArrayList<String>();
    }

    public static Component generateWarningMessage(TrackedStorm storm, WarningLevel level) {
        MutableComponent message = Component.empty();
        message.append(WarningGenerator.getWarningHeader(level));
        message.append((Component)Component.literal((String)"\n"));
        message.append(WarningGenerator.getHazardDescription(storm, level));
        List<String> modifiers = WarningGenerator.getWarningModifiers(storm);
        for (String modifier : modifiers) {
            message.append((Component)Component.literal((String)"\n"));
            message.append((Component)Component.literal((String)modifier).withStyle(ChatFormatting.YELLOW));
        }
        if (level.isTornadoWarning()) {
            message.append((Component)Component.literal((String)"\n"));
            message.append(WarningGenerator.getPrecautionaryActions(level));
        }
        return message;
    }

    private static Component getWarningHeader(WarningLevel level) {
        ChatFormatting color = WarningGenerator.getChatFormatting(level);
        return switch (level) {
            case WarningLevel.TOR_EMERGENCY -> Component.literal((String)"*** TORNADO EMERGENCY ***").withStyle(new ChatFormatting[]{ChatFormatting.BOLD, ChatFormatting.DARK_PURPLE});
            case WarningLevel.TOR_PDS -> Component.literal((String)"*** TORNADO WARNING - PARTICULARLY DANGEROUS SITUATION ***").withStyle(new ChatFormatting[]{ChatFormatting.BOLD, ChatFormatting.LIGHT_PURPLE});
            case WarningLevel.TOR_CONFIRMED -> Component.literal((String)"--TORNADO WARNING--").withStyle(new ChatFormatting[]{ChatFormatting.BOLD, color});
            case WarningLevel.TOR_INDICATED -> Component.literal((String)"--TORNADO WARNING - RADAR INDICATED--").withStyle(new ChatFormatting[]{ChatFormatting.BOLD, color});
            case WarningLevel.SVR_PDS -> Component.literal((String)"--SEVERE THUNDERSTORM WARNING - PDS--").withStyle(new ChatFormatting[]{ChatFormatting.BOLD, ChatFormatting.GOLD});
            case WarningLevel.SVR_DESTRUCTIVE -> Component.literal((String)"--SEVERE THUNDERSTORM WARNING - DESTRUCTIVE--").withStyle(new ChatFormatting[]{ChatFormatting.BOLD, ChatFormatting.GOLD});
            case WarningLevel.SVR_WARNING -> Component.literal((String)"--SEVERE THUNDERSTORM WARNING--").withStyle(new ChatFormatting[]{ChatFormatting.BOLD, ChatFormatting.YELLOW});
            case WarningLevel.STATEMENT -> Component.literal((String)"--SPECIAL WEATHER STATEMENT--").withStyle(new ChatFormatting[]{ChatFormatting.BOLD, ChatFormatting.GRAY});
            default -> Component.empty();
        };
    }

    private static Component getHazardDescription(TrackedStorm storm, WarningLevel level) {
        MutableComponent desc = Component.empty();
        String direction = storm.getMovementDirectionString();
        int speedBPH = storm.getMovementSpeedBlocksPerMCHour();
        int speedMPH = storm.getMovementSpeedMPH();
        if (level.isTornadoWarning()) {
            desc.append((Component)Component.literal((String)String.format("A %s was detected near your area, moving %s at %d MPH / %d BPH.\n", WarningGenerator.getTornadoDescription(level), direction, speedMPH, speedBPH)).withStyle(ChatFormatting.WHITE));
            desc.append((Component)Component.literal((String)"HAZARD: ").withStyle(ChatFormatting.RED));
            desc.append((Component)Component.literal((String)WarningGenerator.getHazardText(storm, level)).withStyle(ChatFormatting.WHITE));
        } else if (level.isSevereThunderstormWarning()) {
            desc.append((Component)Component.literal((String)String.format("A severe thunderstorm was detected near your area, moving %s at %d MPH / %d BPH.\n", direction, speedMPH, speedBPH)).withStyle(ChatFormatting.WHITE));
            String hailSize = WarningGenerator.estimateHailSize(storm);
            desc.append((Component)Component.literal((String)"HAZARD: ").withStyle(ChatFormatting.GOLD));
            desc.append((Component)Component.literal((String)String.format("%s size hail and damaging winds.", hailSize)).withStyle(ChatFormatting.WHITE));
            if (WarningGenerator.shouldMentionTornadoLikely(storm)) {
                desc.append((Component)Component.literal((String)"\nA TORNADO IS LIKELY with this storm.").withStyle(new ChatFormatting[]{ChatFormatting.RED, ChatFormatting.BOLD}));
            } else if (WarningGenerator.shouldMentionTornadoPossible(storm)) {
                desc.append((Component)Component.literal((String)"\nA TORNADO IS POSSIBLE with this storm.").withStyle(new ChatFormatting[]{ChatFormatting.YELLOW, ChatFormatting.BOLD}));
            }
        } else {
            desc.append((Component)Component.literal((String)"Strong thunderstorm activity detected. ").withStyle(ChatFormatting.WHITE));
            desc.append((Component)Component.literal((String)"Small hail and gusty winds possible.").withStyle(ChatFormatting.GRAY));
        }
        return desc;
    }

    private static String getTornadoDescription(WarningLevel level) {
        return switch (level) {
            case WarningLevel.TOR_EMERGENCY -> "VIOLENT TORNADO";
            case WarningLevel.TOR_PDS -> "LARGE AND EXTREMELY DANGEROUS tornado";
            case WarningLevel.TOR_CONFIRMED -> "confirmed tornado";
            case WarningLevel.TOR_INDICATED -> "severe thunderstorm capable of producing a tornado";
            default -> "tornado";
        };
    }

    private static String getHazardText(TrackedStorm storm, WarningLevel level) {
        int windspeed = storm.getWindspeed();
        return switch (level) {
            case WarningLevel.TOR_EMERGENCY -> String.format("Violent, unsurvivable tornado with winds up to %d mph. COMPLETE DESTRUCTION expected.", windspeed);
            case WarningLevel.TOR_PDS -> String.format("Large dangerous tornado with winds up to %d mph. SEVERE DAMAGE EXPECTED.", windspeed);
            case WarningLevel.TOR_CONFIRMED -> String.format("Tornado with winds up to %d mph. Damage to trees and structures expected.", windspeed);
            case WarningLevel.TOR_INDICATED -> "Strong rotation indicated on radar. A tornado may be forming.";
            default -> "Severe weather threat.";
        };
    }

    private static Component getPrecautionaryActions(WarningLevel level) {
        String text = switch (level) {
            case WarningLevel.TOR_EMERGENCY -> "To protect your life, TAKE COVER NOW! This is an EXTREMELY DANGEROUS and LIFE-THREATENING SITUATION!";
            case WarningLevel.TOR_PDS -> "THIS IS A LIFE-THREATENING SITUATION! TAKE COVER NOW! Move immediately to a basement or storm shelter.";
            case WarningLevel.TOR_CONFIRMED, WarningLevel.TOR_INDICATED -> "TAKE COVER NOW! Move to a basement or interior room on the lowest floor of a sturdy building.";
            default -> "Monitor conditions and be prepared to take shelter.";
        };
        ChatFormatting color = level == WarningLevel.TOR_EMERGENCY || level == WarningLevel.TOR_PDS ? ChatFormatting.LIGHT_PURPLE : ChatFormatting.RED;
        return Component.literal((String)text).withStyle(new ChatFormatting[]{color, ChatFormatting.BOLD});
    }

    public static ChatFormatting getChatFormatting(WarningLevel level) {
        return switch (level) {
            case WarningLevel.TOR_EMERGENCY -> ChatFormatting.DARK_PURPLE;
            case WarningLevel.TOR_PDS -> ChatFormatting.LIGHT_PURPLE;
            case WarningLevel.TOR_CONFIRMED, WarningLevel.TOR_INDICATED -> ChatFormatting.RED;
            case WarningLevel.SVR_PDS, WarningLevel.SVR_DESTRUCTIVE -> ChatFormatting.GOLD;
            case WarningLevel.SVR_WARNING -> ChatFormatting.YELLOW;
            case WarningLevel.STATEMENT -> ChatFormatting.GRAY;
            default -> ChatFormatting.WHITE;
        };
    }

    public static Component generateShortNotification(TrackedStorm storm, WarningLevel level) {
        return WarningGenerator.generateChatAlert(storm, level);
    }

    public static Component generateChatAlert(TrackedStorm storm, WarningLevel level) {
        return WarningGenerator.generateChatAlert(storm, level, 0L, 0L);
    }

    public static String generateWarningId(long stormId, WarningLevel level) {
        return "storm_" + stormId + "_" + level.name();
    }

    public static WarningAlertData generateChatAlertWithData(TrackedStorm storm, WarningLevel level, long worldTime, long worldDay) {
        return WarningGenerator.generateChatAlertWithData(storm, level, worldTime, worldDay, true);
    }

    public static WarningAlertData generateChatAlertWithData(TrackedStorm storm, WarningLevel level, long worldTime, long worldDay, boolean isInsidePolygon) {
        String warningText = WarningGenerator.getCompactWarningText(level, isInsidePolygon);
        int color = WarningGenerator.getWarningHexColor(level);
        String warningId = WarningGenerator.generateWarningId(storm.getId(), level);
        String rawTemplate = WarningGenerator.getFilledTemplate(storm, level, worldTime, worldDay);
        MutableComponent warningPart = Component.literal((String)warningText).withStyle(Style.EMPTY.withColor(TextColor.fromRgb((int)color)).withClickEvent(new ClickEvent(ClickEvent.Action.SUGGEST_COMMAND, "wxmc:warning:" + warningId)));
        MutableComponent hoverHint = Component.literal((String)" (Hover for Information)").withStyle(Style.EMPTY.withColor(ChatFormatting.GRAY));
        MutableComponent chatMessage = Component.empty().append((Component)warningPart).append((Component)hoverHint);
        return new WarningAlertData((Component)chatMessage, rawTemplate, color, warningId);
    }

    public static Component generateChatAlert(TrackedStorm storm, WarningLevel level, long worldTime, long worldDay) {
        return WarningGenerator.generateChatAlertWithData((TrackedStorm)storm, (WarningLevel)level, (long)worldTime, (long)worldDay).chatMessage;
    }

    public static String getFilledTemplate(TrackedStorm storm, WarningLevel level, long worldTime, long worldDay) {
        String template = WarningGenerator.getWarningTemplate(storm, level);
        String timeStr = WarningGenerator.formatMinecraftTime(worldTime);
        String expiryTime30Min = WarningGenerator.formatMinecraftTime(worldTime + 500L);
        String expiryTime1Hour = WarningGenerator.formatMinecraftTime(worldTime + 1000L);
        String coords = String.format("(%.0f, %.0f)", storm.getPosition().x, storm.getPosition().z);
        String direction = storm.getMovementDirectionString();
        int velocityBPH = storm.getMovementSpeedBlocksPerMCHour();
        int velocityMPH = storm.getMovementSpeedMPH();
        String source = WarningGenerator.getSourceText(storm, level);
        String threatSource = WarningGenerator.getThreatSourceText(storm, level);
        String tornadoStatus = WarningGenerator.getTornadoStatusText(level);
        String hazard = WarningGenerator.getHazardTextForTemplate(storm, level);
        String hailSize = WarningGenerator.getHailSizeInches(storm);
        int windSpeed = Math.max(storm.getWindspeed(), WarningGenerator.getMinWindForLevel(level));
        template = template.replace("<time>", timeStr);
        template = template.replace("<time + 30 minutes>", expiryTime30Min);
        template = template.replace("<time + 1 hour>", expiryTime1Hour);
        template = template.replace("<day>", String.valueOf(worldDay));
        template = template.replace("<coordinates>", coords);
        template = template.replace("<coordiates>", coords);
        template = template.replace("<direction>", direction);
        template = template.replace("<v>", String.valueOf(velocityBPH));
        template = template.replace("<mph>", String.valueOf(velocityMPH));
        template = template.replace("<source>", source);
        template = template.replace("<threat_source>", threatSource);
        template = template.replace("<tornado_status>", tornadoStatus);
        template = template.replace("<hazard>", hazard);
        template = template.replace("<hail>", hailSize);
        template = template.replace("<wind>", String.valueOf(windSpeed));
        if (WarningGenerator.shouldMentionTornadoPossible(storm) && !level.isTornadoWarning()) {
            template = template.replace("[TORNADO_POSSIBLE_SECTION]", "&fRemain alert for a possible tornado! Tornadoes can develop quickly\n&ffrom severe thunderstorms. If you spot a tornado go at once into the\n&fbasement or small central room in a sturdy structure.");
            template = template.replace("[TORNADO_LINE]", "&fTORNADO...POSSIBLE");
        } else {
            template = template.replace("[TORNADO_POSSIBLE_SECTION]\n\n", "");
            template = template.replace("[TORNADO_POSSIBLE_SECTION]\n", "");
            template = template.replace("[TORNADO_POSSIBLE_SECTION]", "");
            template = template.replace("[TORNADO_LINE]\n", "");
            template = template.replace("[TORNADO_LINE]", "");
        }
        return template;
    }

    private static Component generateNWSHoverText(TrackedStorm storm, WarningLevel level, long worldTime, long worldDay) {
        String filledTemplate = WarningGenerator.getFilledTemplate(storm, level, worldTime, worldDay);
        return WarningGenerator.parseFormattedText(filledTemplate);
    }

    private static String getWarningTemplate(TrackedStorm storm, WarningLevel level) {
        return switch (level) {
            case WarningLevel.STATEMENT -> WarningGenerator.getStatementTemplate();
            case WarningLevel.SVR_WARNING -> WarningGenerator.getSVRWarningTemplate();
            case WarningLevel.SVR_PDS -> WarningGenerator.getSVRConsiderableTemplate();
            case WarningLevel.SVR_DESTRUCTIVE -> WarningGenerator.getSVRDestructiveTemplate();
            case WarningLevel.TOR_INDICATED -> WarningGenerator.getTornadoIndicatedTemplate();
            case WarningLevel.TOR_CONFIRMED -> WarningGenerator.getTornadoObservedTemplate();
            case WarningLevel.TOR_PDS -> WarningGenerator.getTornadoPDSTemplate();
            case WarningLevel.TOR_EMERGENCY -> WarningGenerator.getTornadoEmergencyTemplate();
            default -> WarningGenerator.getStatementTemplate();
        };
    }

    private static String getStatementTemplate() {
        return "#FFE4B5Special Weather Statement\n&aMinecraft Weather Service\n&f<time> - Day <day>\n\n&f...A strong thunderstorm will impact portions of your area through the next 30 minutes...\n\n&fAt <time>, Doppler radar was tracking a strong thunderstorm near <coordinates>, moving <direction> at <mph> MPH / <v> BPH.\n\n&fHAZARD...&fWinds in excess of <wind> and <hail> size hail.\n\n&fSOURCE...&fRadar indicated.\n\n&fIMPACT...&fGusty winds could knock down tree limbs and blow around unsecured objects. Minor damage to outdoor objects is possible.\n\n&f&nPRECAUTIONARY/PREPAREDNESS ACTIONS...&r\n\n&fIf outdoors, consider seeking shelter inside a building.\n\n&fMAX HAIL SIZE...&f<hail>\n&fMAX WIND GUST...&f<wind> MPH";
    }

    private static String getSVRWarningTemplate() {
        return "#FFA500Severe Thunderstorm Warning\n&aMinecraft Weather Service\n&f<time> - Day <day>\n\n&fThe Minecraft Weather Service has issued a\n\n&f* Severe Thunderstorm Warning for...\n  &f<coordinates>\n\n&f* Until <time + 1 hour>.\n\n&f* At <time>, a severe thunderstorm was located near <coordinates>,\n  &fmoving <direction> at <mph> MPH / <v> BPH.\n\n  &fHAZARD...&f<hazard>\n\n  &fSOURCE...&f<source>\n\n  &fIMPACT...&fExpect damage to roofs, siding, and trees.\n\n&f&nPRECAUTIONARY/PREPAREDNESS ACTIONS...&r\n\n[TORNADO_POSSIBLE_SECTION]\n\n&fFor your protection move to an interior room on the lowest floor of a\n&fbuilding.\n\n&f-\n\n[TORNADO_LINE]\n&fHAIL THREAT...&f<threat_source>\n&fMAX HAIL SIZE...&f<hail>\n&fWIND THREAT...&f<threat_source>\n&fMAX WIND GUST...&f<wind> MPH";
    }

    private static String getSVRConsiderableTemplate() {
        return "#FFA500Severe Thunderstorm Warning\n&aMinecraft Weather Service\n&f<time> - Day <day>\n\n&fThe Minecraft Weather Service has issued a\n\n&f* Severe Thunderstorm Warning for...\n  &f<coordinates>\n\n&f* Until <time + 1 hour>.\n\n&f* At <time>, a severe thunderstorm was located near <coordinates>,\n  &fmoving <direction> at <mph> MPH / <v> BPH.\n\n  &fHAZARD...&f<hazard>\n\n  &fSOURCE...&f<source>\n\n  &fIMPACT...&fPeople and animals outdoors will be injured. Expect hail\n            &fdamage to roofs, siding, windows, and vehicles. Expect\n            &fwind damage to trees and powerlines.\n\n&f&nPRECAUTIONARY/PREPAREDNESS ACTIONS...&r\n\n[TORNADO_POSSIBLE_SECTION]\n\n&fFor your protection move to an interior room on the lowest floor of a\n&fbuilding.\n\n&f-\n\n[TORNADO_LINE]\n&fTHUNDERSTORM DAMAGE THREAT...&fCONSIDERABLE\n&fHAIL THREAT...&f<threat_source>\n&fMAX HAIL SIZE...&f<hail>\n&fWIND THREAT...&f<threat_source>\n&fMAX WIND GUST...&f<wind> MPH";
    }

    private static String getSVRDestructiveTemplate() {
        return "#FFA500Severe Thunderstorm Warning\n&aMinecraft Weather Service\n&f<time> - Day <day>\n\n&fThe Minecraft Weather Service has issued a\n\n&f* Severe Thunderstorm Warning for...\n  &f<coordinates>\n\n&f* Until <time + 1 hour>.\n\n&f* At <time>, a severe thunderstorm was located near <coordinates>,\n  &fmoving <direction> at <mph> MPH / <v> BPH.\n\n  &c&lTHIS IS A DESTRUCTIVE STORM!\n\n  &fHAZARD...&f<hazard>\n\n  &fSOURCE...&f<source>\n\n  &fIMPACT...&fPeople and animals outdoors will be severely injured.\n            &fExpect shattered windows, extensive damage to roofs,\n            &fsiding, and vehicles.\n\n&f&nPRECAUTIONARY/PREPAREDNESS ACTIONS...&r\n\n[TORNADO_POSSIBLE_SECTION]\n\n&fPrepare immediately for large hail and/or damaging winds. People outside\n&fshould move immediately to shelter inside a strong building. Stay\n&faway from windows.\n\n&f-\n\n[TORNADO_LINE]\n&fTHUNDERSTORM DAMAGE THREAT...&fDESTRUCTIVE\n&fHAIL THREAT...&f<threat_source>\n&fMAX HAIL SIZE...&f<hail>\n&fWIND THREAT...&f<threat_source>\n&fMAX WIND GUST...&f<wind> MPH";
    }

    private static String getTornadoIndicatedTemplate() {
        return "#FF0000Tornado Warning\n&aMinecraft Weather Service\n&f<time> - Day <day>\n\n&fThe Minecraft Weather Service has issued a\n\n&f* Tornado Warning for...\n  &f<coordinates>\n\n&f* Until <time + 1 hour>.\n\n&f* At <time>, a severe thunderstorm capable of producing a tornado was located near <coordinates>,\n  &fmoving <direction> at <mph> MPH / <v> BPH.\n\n  &fHAZARD...&fTornado\n\n  &fSOURCE...&f<source>\n\n  &fIMPACT...&fFlying debris will be dangerous to those caught without\n            &fshelter. Mobile homes will be damaged or destroyed.\n            &fDamage to roofs, windows, and vehicles will occur.\n            &fTree damage is likely.\n\n&f&nPRECAUTIONARY/PREPAREDNESS ACTIONS...&r\n\n&fTAKE COVER NOW! Move to a basement or an interior room on the lowest\n&ffloor of a sturdy building. Avoid windows. If you are outdoors, in a\n&fmobile home, or in a vehicle, move to the closest substantial shelter\n&fand protect yourself from flying debris.\n\n&f-\n\n&fTORNADO...&f<tornado_status>\n&fMAX HAIL SIZE...&f<hail>";
    }

    private static String getTornadoObservedTemplate() {
        return "#FF0000Tornado Warning\n&aMinecraft Weather Service\n&f<time> - Day <day>\n\n&fThe Minecraft Weather Service has issued a\n\n&f* Tornado Warning for...\n  &f<coordinates>\n\n&f* Until <time + 1 hour>.\n\n&f* At <time>, a confirmed tornado was located near <coordinates>,\n  &fmoving <direction> at <mph> MPH / <v> BPH.\n\n  &fHAZARD...&fTornado\n\n  &fSOURCE...&f<source>\n\n  &fIMPACT...&fFlying debris will be dangerous to those caught without\n            &fshelter. Mobile homes will be damaged or destroyed.\n            &fDamage to roofs, windows, and vehicles will occur.\n            &fTree damage is likely.\n\n&f&nPRECAUTIONARY/PREPAREDNESS ACTIONS...&r\n\n&fTo repeat, a tornado is on the ground. TAKE COVER NOW! Move to a\n&fbasement or an interior room on the lowest floor of a sturdy\n&fbuilding. Avoid windows. If you are outdoors, in a mobile home, or in\n&fa vehicle, move to the closest substantial shelter and protect\n&fyourself from flying debris.\n\n&f-\n\n&fTORNADO...&f<tornado_status>\n&fMAX HAIL SIZE...&f<hail>";
    }

    private static String getTornadoPDSTemplate() {
        return "#FF0000Tornado Warning\n&aMinecraft Weather Service\n&f<time> - Day <day>\n\n&fThe Minecraft Weather Service has issued a\n\n&f* Tornado Warning for...\n  &f<coordinates>\n\n&f* Until <time + 1 hour>.\n\n&f* At <time>, a large and extremely dangerous tornado was located near <coordinates>,\n  &fmoving <direction> at <mph> MPH / <v> BPH.\n\n  &c&lThis is a PARTICULARLY DANGEROUS SITUATION. TAKE COVER NOW!\n\n  &fHAZARD...&fTornado\n\n  &fSOURCE...&f<source>\n\n  &fIMPACT...&fYou are in a life-threatening situation. Flying debris\n            &fmay be deadly to those caught without shelter. Mobile\n            &fhomes will be destroyed. Considerable damage to homes,\n            &fbusinesses, and vehicles is likely and complete\n            &fdestruction is possible.\n\n&f&nPRECAUTIONARY/PREPAREDNESS ACTIONS...&r\n\n&fTo repeat, a large, extremely dangerous and potentially deadly\n&ftornado is developing. To protect your life, TAKE COVER NOW! Move to\n&fa basement or an interior room on the lowest floor of a sturdy\n&fbuilding. Avoid windows. If you are outdoors or in a vehicle, move\n&fto the closest substantial shelter and protect yourself from flying\n&fdebris.\n\n&f-\n\n&fTORNADO...&f<tornado_status>\n&fTORNADO DAMAGE THREAT...&fCONSIDERABLE\n&fMAX HAIL SIZE...&f<hail>";
    }

    private static String getTornadoEmergencyTemplate() {
        return "#FF0000Tornado Warning\n&aMinecraft Weather Service\n&f<time> - Day <day>\n\n...TORNADO EMERGENCY FOR <coordinates>...\n\n&fThe Minecraft Weather Service has issued a\n\n&f* Tornado Warning for...\n  &f<coordinates>\n\n&f* Until <time + 1 hour>.\n\n&f* At <time>, a large and extremely dangerous tornado was located near <coordinates>,\n  &fmoving <direction> at <mph> MPH / <v> BPH.\n\n  &c&lTORNADO EMERGENCY for <coordinates>.\n  &c&lThis is a PARTICULARLY DANGEROUS SITUATION. TAKE COVER NOW!\n\n  &fHAZARD...&fDeadly Tornado\n\n  &fSOURCE...&f<source>\n\n  &fIMPACT...&fYou are in a life-threatening situation. Flying debris\n            &fmay be deadly to those caught without shelter. Mobile\n            &fhomes will be destroyed. Considerable damage to homes,\n            &fbusinesses, and vehicles is likely and complete\n            &fdestruction is possible.\n\n&f&nPRECAUTIONARY/PREPAREDNESS ACTIONS...&r\n\n&fTo repeat, a large, extremely dangerous and potentially deadly\n&ftornado is on the ground. To protect your life, TAKE COVER NOW! Move\n&fto an interior room on the lowest floor of a sturdy building. Avoid\n&fwindows. If in a mobile home, a vehicle or outdoors, move to the\n&fclosest substantial shelter and protect yourself from flying debris.\n\n&f-\n\n&fTORNADO...&f<tornado_status>\n&fTORNADO DAMAGE THREAT...&fCATASTROPHIC\n&fMAX HAIL SIZE...&f<hail>";
    }

    private static Component parseFormattedText(String text) {
        MutableComponent result = Component.empty();
        Style currentStyle = Style.EMPTY;
        StringBuilder currentText = new StringBuilder();
        int i = 0;
        block26: while (i < text.length()) {
            char c = text.charAt(i);
            if (c == '#' && i + 6 < text.length()) {
                if (currentText.length() > 0) {
                    result.append((Component)Component.literal((String)currentText.toString()).withStyle(currentStyle));
                    currentText = new StringBuilder();
                }
                String hexCode = text.substring(i + 1, i + 7);
                try {
                    int color = Integer.parseInt(hexCode, 16);
                    currentStyle = Style.EMPTY.withColor(TextColor.fromRgb((int)color));
                    i += 7;
                    continue;
                }
                catch (NumberFormatException numberFormatException) {
                    // empty catch block
                }
            }
            if (c == '&' && i + 1 < text.length()) {
                char code = text.charAt(i + 1);
                if (currentText.length() > 0) {
                    result.append((Component)Component.literal((String)currentText.toString()).withStyle(currentStyle));
                    currentText = new StringBuilder();
                }
                switch (code) {
                    case '0': {
                        currentStyle = currentStyle.withColor(ChatFormatting.BLACK);
                        break;
                    }
                    case '1': {
                        currentStyle = currentStyle.withColor(ChatFormatting.DARK_BLUE);
                        break;
                    }
                    case '2': {
                        currentStyle = currentStyle.withColor(ChatFormatting.DARK_GREEN);
                        break;
                    }
                    case '3': {
                        currentStyle = currentStyle.withColor(ChatFormatting.DARK_AQUA);
                        break;
                    }
                    case '4': {
                        currentStyle = currentStyle.withColor(ChatFormatting.DARK_RED);
                        break;
                    }
                    case '5': {
                        currentStyle = currentStyle.withColor(ChatFormatting.DARK_PURPLE);
                        break;
                    }
                    case '6': {
                        currentStyle = currentStyle.withColor(ChatFormatting.GOLD);
                        break;
                    }
                    case '7': {
                        currentStyle = currentStyle.withColor(ChatFormatting.GRAY);
                        break;
                    }
                    case '8': {
                        currentStyle = currentStyle.withColor(ChatFormatting.DARK_GRAY);
                        break;
                    }
                    case '9': {
                        currentStyle = currentStyle.withColor(ChatFormatting.BLUE);
                        break;
                    }
                    case 'a': {
                        currentStyle = currentStyle.withColor(ChatFormatting.GREEN);
                        break;
                    }
                    case 'b': {
                        currentStyle = currentStyle.withColor(ChatFormatting.AQUA);
                        break;
                    }
                    case 'c': {
                        currentStyle = currentStyle.withColor(ChatFormatting.RED);
                        break;
                    }
                    case 'd': {
                        currentStyle = currentStyle.withColor(ChatFormatting.LIGHT_PURPLE);
                        break;
                    }
                    case 'e': {
                        currentStyle = currentStyle.withColor(ChatFormatting.YELLOW);
                        break;
                    }
                    case 'f': {
                        currentStyle = currentStyle.withColor(ChatFormatting.WHITE);
                        break;
                    }
                    case 'l': {
                        currentStyle = currentStyle.withBold(Boolean.valueOf(true));
                        break;
                    }
                    case 'n': {
                        currentStyle = currentStyle.withUnderlined(Boolean.valueOf(true));
                        break;
                    }
                    case 'o': {
                        currentStyle = currentStyle.withItalic(Boolean.valueOf(true));
                        break;
                    }
                    case 'm': {
                        currentStyle = currentStyle.withStrikethrough(Boolean.valueOf(true));
                        break;
                    }
                    case 'k': {
                        currentStyle = currentStyle.withObfuscated(Boolean.valueOf(true));
                        break;
                    }
                    case 'r': {
                        currentStyle = Style.EMPTY;
                        break;
                    }
                    default: {
                        currentText.append(c);
                        ++i;
                        continue block26;
                    }
                }
                i += 2;
                continue;
            }
            currentText.append(c);
            ++i;
        }
        if (currentText.length() > 0) {
            result.append((Component)Component.literal((String)currentText.toString()).withStyle(currentStyle));
        }
        return result;
    }

    private static String formatMinecraftTime(long worldTime) {
        String ampm;
        long adjustedTime = (worldTime + 6000L) % 24000L;
        int hours = (int)(adjustedTime / 1000L);
        int minutes = (int)(adjustedTime % 1000L * 60L / 1000L);
        String string = ampm = hours >= 12 ? "PM" : "AM";
        if ((hours %= 12) == 0) {
            hours = 12;
        }
        return String.format("%d:%02d %s", hours, minutes, ampm);
    }

    private static String getSourceText(TrackedStorm storm, WarningLevel level) {
        if (level == WarningLevel.TOR_CONFIRMED || level == WarningLevel.TOR_PDS || level == WarningLevel.TOR_EMERGENCY) {
            if (storm.isSpotterConfirmed()) {
                return "Weather spotters confirmed tornado.";
            }
            return "Radar confirmed tornado.";
        }
        if (level == WarningLevel.TOR_INDICATED) {
            if (storm.isSpotterConfirmed()) {
                return "Weather spotters confirmed tornado.";
            }
            return "Radar indicated rotation.";
        }
        if (storm.isSpotterReportedHail() || storm.isSpotterReportedWind()) {
            return "Trained weather spotters.";
        }
        return "Radar indicated.";
    }

    private static String getThreatSourceText(TrackedStorm storm, WarningLevel level) {
        if (storm.isSpotterReportedHail() || storm.isSpotterReportedWind()) {
            return "OBSERVED";
        }
        return "RADAR INDICATED";
    }

    private static String getHailThreatSourceText(TrackedStorm storm) {
        if (storm.isSpotterReportedHail()) {
            return "OBSERVED";
        }
        return "RADAR INDICATED";
    }

    private static String getWindThreatSourceText(TrackedStorm storm) {
        if (storm.isSpotterReportedWind()) {
            return "OBSERVED";
        }
        return "RADAR INDICATED";
    }

    private static String getTornadoStatusText(WarningLevel level) {
        if (level == WarningLevel.TOR_CONFIRMED || level == WarningLevel.TOR_PDS || level == WarningLevel.TOR_EMERGENCY) {
            return "OBSERVED";
        }
        return "RADAR INDICATED";
    }

    private static String getHazardTextForTemplate(TrackedStorm storm, WarningLevel level) {
        int wind = Math.max(storm.getWindspeed(), WarningGenerator.getMinWindForLevel(level));
        String hail = WarningGenerator.estimateHailSize(storm);
        return switch (level) {
            case WarningLevel.STATEMENT -> String.format("Winds in excess of %d mph and penny size hail", wind);
            case WarningLevel.SVR_WARNING -> String.format("%d mph wind gusts and %s size hail", wind, hail.toLowerCase());
            case WarningLevel.SVR_PDS -> String.format("%d mph wind gusts and %s size hail", wind, hail.toLowerCase());
            case WarningLevel.SVR_DESTRUCTIVE -> String.format("%d mph wind gusts and %s size hail", wind, hail.toLowerCase());
            case WarningLevel.TOR_INDICATED -> "Tornado";
            case WarningLevel.TOR_EMERGENCY, WarningLevel.TOR_PDS, WarningLevel.TOR_CONFIRMED -> "Damaging tornado.";
            default -> "Severe weather";
        };
    }

    private static int getMinWindForLevel(WarningLevel level) {
        return switch (level) {
            case WarningLevel.STATEMENT -> 30;
            case WarningLevel.SVR_WARNING -> 60;
            case WarningLevel.SVR_PDS -> 70;
            case WarningLevel.SVR_DESTRUCTIVE -> 80;
            default -> 0;
        };
    }

    private static String getHailSizeInches(TrackedStorm storm) {
        String size;
        return switch (size = WarningGenerator.estimateHailSize(storm)) {
            case "Pea to marble" -> "0.25\"";
            case "Penny to nickel" -> "0.75\"";
            case "Quarter" -> "1.00\"";
            case "Golf ball" -> "1.75\"";
            case "Tennis ball" -> "2.50\"";
            case "Baseball or larger" -> "2.75+\"";
            default -> "0.50\"";
        };
    }

    public static String getCompactWarningText(WarningLevel level, boolean isInsidePolygon) {
        String locationText = isInsidePolygon ? "for your location" : "near your location";
        return switch (level) {
            case WarningLevel.STATEMENT -> "A Special Weather Statement has been issued " + locationText + ".";
            case WarningLevel.SVR_WARNING -> "A Severe Thunderstorm Warning has been issued " + locationText + ".";
            case WarningLevel.SVR_PDS -> "A Severe Thunderstorm Warning -CONSIDERABLE- has been issued " + locationText + ".";
            case WarningLevel.SVR_DESTRUCTIVE -> "A Severe Thunderstorm Warning -DESTRUCTIVE- has been issued " + locationText + ".";
            case WarningLevel.TOR_INDICATED -> "A Tornado Warning -RADAR INDICATED- has been issued " + locationText + ".";
            case WarningLevel.TOR_CONFIRMED -> "A Tornado Warning -OBSERVED- has been issued " + locationText + ".";
            case WarningLevel.TOR_PDS -> "A Tornado Warning -PDS- has been issued " + locationText + ".";
            case WarningLevel.TOR_EMERGENCY -> "A Tornado Emergency has been issued " + locationText + ".";
            default -> "Weather alert issued " + locationText + ".";
        };
    }

    public static String getCompactWarningText(WarningLevel level) {
        return WarningGenerator.getCompactWarningText(level, true);
    }

    public static int getWarningHexColor(WarningLevel level) {
        return switch (level) {
            case WarningLevel.STATEMENT -> 16770229;
            case WarningLevel.SVR_PDS, WarningLevel.SVR_DESTRUCTIVE, WarningLevel.SVR_WARNING -> 16753920;
            case WarningLevel.TOR_EMERGENCY, WarningLevel.TOR_PDS, WarningLevel.TOR_CONFIRMED, WarningLevel.TOR_INDICATED -> 0xFF0000;
            default -> 0xFFFFFF;
        };
    }

    public static class WarningAlertData {
        public final Component chatMessage;
        public final String rawTemplate;
        public final int color;
        public final String warningId;

        public WarningAlertData(Component chatMessage, String rawTemplate, int color, String warningId) {
            this.chatMessage = chatMessage;
            this.rawTemplate = rawTemplate;
            this.color = color;
            this.warningId = warningId;
        }
    }
}

